
#ifndef CRYS_RND_H
#define CRYS_RND_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
#include "DX_VOS_BaseTypes.h"
#include "CRYS_error.h"
#include "CRYS_Defs.h"
#include "LLF_RND_EngineInfo.h"

#ifdef __cplusplus
extern "C"
{
#endif

  /*
   *  Object CRYS_RND.h
   *  State           :  %state%
   *  Creation date   :  Thu Dec 16 17:11:32 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This file contains the CRYS APIs used for random numbergeneration.
   *
   *  \version CRYS_RND.h#1:incl:15
   *  \author adams
   *  \remarks Copyright (C) 2005 by Discretix Technologies Ltd.
   *           All Rights reserved
   */




/************************ Defines ******************************/

/** 
@brief - a definition describing the low level Engine type ( SW , Hardware , Etc )
*/

#define CRYS_RND_ENGINE_TYPE  LLF_RND_ENGINE_TYPE

/************************ Enums ********************************/


/************************ Typedefs  ****************************/


/************************ Structs  ******************************/


/************************ Public Variables **********************/


/************************ Public Functions **********************/
/* -----------------------------------------------------------------------------
 * @brief The CRYS_RND_Instantiation initialize the RNG. Its required before any
 *					output can be produced. It is initated in the CRYS_RND_Init phase.
 *					This function must be called at least once per system reset.
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CIMPORT_C CRYSError_t CRYS_RND_Instantiation(void);


/* ------------------------------------------------------------
 * @brief The CRYS_RND_GenerateVector function generates a random vector,
 *			 using the FIPS-PUB [SP800-90].
 *
 *        The random function is based on the AES. The code is written according to the:
 *			 CryptoCore: Random Number Geneators System Specification (Revision 1.3)
 *
 *
 * @RndSize[in] - The size of random bytes that is required.
 *
 * @Output_ptr[in,out] The output vector.
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CIMPORT_C CRYSError_t CRYS_RND_GenerateVector(DxUint16_t RndSize, DxUint8_t *Output_ptr);


/* ------------------------------------------------------------/
 * @brief The CRYS_RND_ResetSeed is used for:
 * 			 1. Mixing of additonal entropy into the working state.
 *			 2. Mixing additional input provided by the user called additional input buffer.        
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CIMPORT_C CRYSError_t CRYS_RND_Reseeding( void );

/**********************************************************************************************************/
/* 
 * @brief The CRYS_RND_GenerateVectorInRange function generates a random vector Rand in range:
 *            1 < RandVect < MaxVect:
 *        The function performs the following: 
 *        1.  Calls the CRYS_RND_GenerateVector() function for generating random vector 
 *            RndVect of size in bytes.
 *        2.  Zeroes extra bits of RndVect.
 *        3.  Compares RndVect with 1 and MaxVect. If condition  1 < RandVect < MaxVect
 *            is not satisfied, then generates new random.
 *
 * @RndSize[in] - The size of random vectore that is required.
 * @MaxVect_ptr[in]  - The pointer to vector defines high limit of random vector.
 * @RndVect_ptr[in,out] The output vector.
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CIMPORT_C CRYSError_t CRYS_RND_GenerateVectorInRange(DxUint32_t  RndSizeInBits, 
											         DxUint8_t *MaxVect_ptr, DxUint8_t *RndVect_ptr );
											         
											         
/*************************************************************************************/
/**
 * @brief The CRYS_RND_AddAdditionalInput is used for:
 *		  Add additional input by the user.
 *		  This data is used to input additional data for random generation
 *	      reseeding operation or Instantiation.
 *		  The size must be 16 or 32 bytes.
 *
 * @AdditonalInput_ptr[in] - The AdditionalInput buffer.
 * 
 * AdditonalInputSize[in] - The size of the AdditionalInput buffer, can be 16 or 32 bytes.
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CIMPORT_C CRYSError_t CRYS_RND_AddAdditionalInput( DxUint8_t *AdditonalInput_ptr,
													DxUint16_t AdditonalInputSize);
													
/**************************************************************************************/
/**
 * @brief The CRYS_RND_ResetSeed resets the SEED generated by the low level 
 *        ( hardware or OPerating system service on software ).  
 * This is Dummy function for backward compatbility
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CEXPORT_C CRYSError_t CRYS_RND_ResetSeed( void );


/**********************************************************************************************************/
/**
 * @brief The CRYS_RND_StartResetSeed start the hardware bit random bit collection for the seeding/instansiation 
 *        opeartion)
 * 
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* as defined in ...
 */
CEXPORT_C CRYSError_t CRYS_RND_StartResetSeed( void );												
#ifdef __cplusplus
}
#endif

#endif
